"use client";
import React, { useEffect, useState } from "react";
import { FontAwesomeIcon } from "@fortawesome/react-fontawesome";
import { faPen, faTrash } from "@fortawesome/free-solid-svg-icons";
import { useFormik } from "formik";
import * as Yup from "yup";
import toast from "react-hot-toast";
import {
  DeleteRequest,
  PostRequest,
} from "@/app/components/axiosConfig/axiosConfig";
import ExperienceModal from "../ExperienceModal";
import { useGetExperienceQuery } from "@/services/Api";
import { useRouter } from "next/navigation";

const Education = () => {

  const [loading, setLoading] = useState(false);
  const [allExperience, setAllExperience] = useState([]);
  const [isModalOpen, setIsModalOpen] = useState(false);
  const [selectedExperience, setSelectedExperience] = useState(null);
  const router = useRouter();

  const {
    data: getExperience,
    refetch: getExperienceRefetch,
  } = useGetExperienceQuery();

  useEffect(() => {
    setAllExperience(getExperience?.experiance);
  }, [getExperience]);

  useEffect(() => {
    getExperienceRefetch();
  }, []);

  const formik = useFormik({
    initialValues: {
      company_name: "",
      designation: "",
      start_date: "",
      end_date: "",
    },
    validationSchema: Yup.object({
      company_name: Yup.string().required("Company Name is required"),
      designation: Yup.string().required("Field of designation is required"),
      start_date: Yup.date().required("Start Date is required"),
      end_date: Yup.date().required("End Date is required"),
    }),
    onSubmit: async (values, { setSubmitting, resetForm }) => {
      const authToken = localStorage.getItem("authToken");
      try {
        const res = await PostRequest("/experiance", values, authToken);
        toast.success("Experience added successfully!", {
          position: "top-center",
        });
        getExperienceRefetch();
        resetForm();
        router.push("/profile");
      } catch (error) {
        console.error("Update education Error", error);
        toast.error("Experience Not Added!", {
          position: "top-center",
        });
      } finally {
        setLoading(false);
        setSubmitting(false);
      }
    },
  });

  const handleDeleteExperience = async (id) => {
    const confirmDelete = window.confirm(
      "Are you sure you want to delete this experience?"
    );

    if (confirmDelete) {
      const authToken = localStorage.getItem("authToken");
      try {
        const res = await DeleteRequest(`/experiance/${id}`, authToken);

        toast.success("Experience deleted successfully!", {
          position: "top-center",
        });

        getExperienceRefetch();
      } catch (error) {
        console.error("Failed to delete experience:", error);
        toast.error("Failed to delete experience", {
          position: "top-center",
        });
      }
    }
  };

  const handleEditExperience = (experience) => {
    setSelectedExperience(experience);
    setIsModalOpen(true);
  };

  const handleModalSubmit = async (updatedExperience) => {
    setLoading(true);
    const authToken = localStorage.getItem("authToken");
    try {
      await PostRequest(
        `/experiance/${selectedExperience.id}`,
        updatedExperience,
        authToken
      );
      toast.success("Experience updated successfully!", {
        position: "top-center",
      });
      getExperienceRefetch();
      setIsModalOpen(false);
    } catch (error) {
      console.error("Failed to update Experience:", error);
      toast.error("Failed to update Experience.", { position: "top-center" });
    } finally {
      setLoading(false);
    }
  };

  return (
    <div className="min-h-screen md:px-10 px-4 py-6">
      <div className="bg-[#FFFAFA] min-h-[95vh] rounded-2xl p-4 lg:p-8">
        <div className="text-3xl font-bold">Experience</div>
        <div className="md:w-2/3 mt-10">
          {allExperience?.length ? (
            <div>
              {allExperience?.map((item, index) => (
                <div key={index} className="flex flex-col gap-4 mb-8">
                  <div className="flex justify-between items-center">
                    <div className="text-2xl text-golden font-bold">
                      Experience
                    </div>
                    <div className="flex gap-4">
                      <button
                        onClick={() => handleDeleteExperience(item.id)}
                      >
                        <FontAwesomeIcon
                          icon={faTrash}
                          className="text-golden w-4 h-4"
                        />
                      </button>
                      <button
                        onClick={() => handleEditExperience(item)}
                      >
                        <FontAwesomeIcon
                          icon={faPen}
                          className="text-golden w-4 h-4"
                        />
                      </button>
                    </div>
                  </div>
                  <div className="flex gap-2 justify-between">
                    <div className="flex gap-1 flex-col items-start">
                      <span className="text-lg font-bold">Company Name</span>
                      <span className="text-base text-[#0000005C] font-medium">
                        {item?.company_name}
                      </span>
                    </div>
                    <div className="min-w-36 flex gap-1 items-start flex-col">
                      <span className="text-lg font-bold">Designation</span>
                      <span className="text-base text-[#0000005C] font-medium">
                        {item?.designation}
                      </span>
                    </div>
                  </div>
                  <div className="flex gap-2 justify-between">
                    <div className="flex gap-1 items-start flex-col">
                      <span className="text-lg font-bold">Start Date</span>
                      <span className="text-base text-[#0000005C] font-medium">
                        {item?.start_date}
                      </span>
                    </div>
                    <div className="min-w-36 flex gap-1 items-start  flex-col">
                      <span className="text-lg font-bold">End Date</span>
                      <span className="text-base text-[#0000005C] font-medium">
                        {item?.end_date}
                      </span>
                    </div>
                  </div>
                </div>
              ))}
            </div>
          ) : (
            <div>No experience data available.</div>
          )}
        </div>
        <div className="mt-8">
          <form onSubmit={formik.handleSubmit}>
            <div className="flex flex-col-reverse md:flex-row gap-6 justify-between">
              <div className="md:w-2/3 flex flex-col gap-6">
                <div className="text-2xl text-golden font-bold">
                  Add New Experience
                </div>

                <div className="flex gap-2 flex-col">
                  <label className="text-[16px] font-bold">Company Name</label>
                  <input
                    type="text"
                    name="company_name"
                    value={formik.values.company_name}
                    onChange={formik.handleChange}
                    onBlur={formik.handleBlur}
                    placeholder="Enter company name"
                    className={`outline-none border rounded-lg px-3 py-3 bg-[#FFFAFA] ${formik.touched.company_name && formik.errors.company_name
                      ? "border-red-500"
                      : "border-gray-700 focus:border-golden"
                      }`}
                  />
                  {formik.touched.company_name && formik.errors.company_name ? (
                    <div className="text-red-500 text-sm">
                      {formik.errors.company_name}
                    </div>
                  ) : null}
                </div>

                <div className="flex gap-2 flex-col">
                  <label className="text-[16px] font-bold">Designation</label>
                  <input
                    type="text"
                    name="designation"
                    value={formik.values.designation}
                    onChange={formik.handleChange}
                    onBlur={formik.handleBlur}
                    placeholder="Enter designation"
                    className={`outline-none border rounded-lg px-3 py-3 bg-[#FFFAFA] ${formik.touched.designation && formik.errors.designation
                      ? "border-red-500"
                      : "border-gray-700 focus:border-golden"
                      }`}
                  />
                  {formik.touched.designation && formik.errors.designation ? (
                    <div className="text-red-500 text-sm">
                      {formik.errors.designation}
                    </div>
                  ) : null}
                </div>

                <div className="flex gap-4">
                  <div className="flex gap-2 flex-col w-1/2">
                    <label className="text-[16px] font-bold">Start Date</label>
                    <input
                      type="date"
                      name="start_date"
                      value={formik.values.start_date}
                      onChange={formik.handleChange}
                      onBlur={formik.handleBlur}
                      placeholder="Enter Start Date"
                      className={`outline-none border rounded-lg px-3 py-3 bg-[#FFFAFA] ${formik.touched.start_date && formik.errors.start_date
                        ? "border-red-500"
                        : "border-gray-700 focus:border-golden"
                        }`}
                    />
                    {formik.touched.start_date && formik.errors.start_date ? (
                      <div className="text-red-500 text-sm">
                        {formik.errors.start_date}
                      </div>
                    ) : null}
                  </div>
                  <div className="flex gap-2 flex-col w-1/2">
                    <label className="text-[16px] font-bold">End Date</label>
                    <input
                      type="date"
                      name="end_date"
                      value={formik.values.end_date}
                      onChange={formik.handleChange}
                      onBlur={formik.handleBlur}
                      placeholder="Enter End Date"
                      className={`outline-none border rounded-lg px-3 py-3 bg-[#FFFAFA] ${formik.touched.end_date && formik.errors.end_date
                        ? "border-red-500"
                        : "border-gray-700 focus:border-golden"
                        }`}
                    />
                    {formik.touched.end_date && formik.errors.end_date ? (
                      <div className="text-red-500 text-sm">
                        {formik.errors.end_date}
                      </div>
                    ) : null}
                  </div>
                </div>
                <div className="flex justify-end mt-6">
                  <button
                    type="submit"
                    className=" bg-golden rounded-xl px-6 text-white font-medium py-2"
                    disabled={loading}
                  >
                    {loading ? "loading..." : "Add Experience"}
                  </button>
                </div>
              </div>
            </div>
          </form>
        </div>
      </div>
      {isModalOpen && (
        <div>
          <ExperienceModal
            isOpen={isModalOpen}
            onClose={() => setIsModalOpen(false)}
            onSubmit={handleModalSubmit}
            selectedExperience={selectedExperience}
          />
        </div>
      )}
    </div>
  );
};

export default Education;